<?php

namespace wbb\system\user\notification\event;

use wbb\data\board\Board;
use wbb\data\board\BoardAction;
use wbb\data\board\BoardCache;
use wbb\data\post\Post;
use wbb\data\post\PostAction;
use wbb\data\thread\Thread;
use wbb\data\thread\ThreadAction;
use wbb\system\cache\builder\BoardCacheBuilder;
use wbb\system\cache\builder\BoardDataCacheBuilder;
use wbb\system\log\modification\PostModificationLogHandler;
use wbb\system\log\modification\ThreadModificationLogHandler;
use wcf\data\modification\log\ModificationLog;
use wcf\data\object\type\ObjectTypeCache;
use wcf\data\user\UserProfile;
use wcf\system\user\notification\TestableUserNotificationEventHandler;
use wcf\system\WCF;

/**
 * Provides methods to create boards, threads and posts for testing user notification events.
 *
 * @author  Matthias Schmidt
 * @copyright   2001-2019 WoltLab GmbH
 * @license WoltLab License <http://www.woltlab.com/license-agreement.html>
 * @package WoltLabSuite\Forum\System\User\Notification\Event
 * @since   3.1
 */
trait TTestablePostRelatedUserNotificationEvent
{
    /**
     * Creates a board for testing.
     *
     * @return  Board
     */
    public static function createTestBoard()
    {
        $board = (new BoardAction([], 'create', [
            'data' => [
                'boardType' => Board::TYPE_BOARD,
                'description' => 'Test Board Description',
                'descriptionUseHtml' => 0,
                'parentID' => null,
                'position' => 1,
                'title' => 'Test Board Title',
            ],
        ]))->executeAction()['returnValues'];

        // reset caches
        TestableUserNotificationEventHandler::getInstance()->resetCacheBuilder(BoardCacheBuilder::getInstance());
        TestableUserNotificationEventHandler::getInstance()->resetCacheBuilder(BoardDataCacheBuilder::getInstance());

        $reflectionClass = new \ReflectionClass(BoardCache::class);
        $reflectionMethod = $reflectionClass->getMethod('init');
        $reflectionMethod->setAccessible(true);
        $reflectionMethod->invoke(BoardCache::getInstance());

        return $board;
    }

    /**
     * Creates a post for testing.
     *
     * @param   UserProfile $threadAuthor
     * @param   UserProfile $postAuthor
     * @return  Post
     */
    public static function createTestPost(UserProfile $threadAuthor, UserProfile $postAuthor)
    {
        return (new PostAction([], 'create', [
            'data' => [
                'message' => 'Test Post Message',
                'threadID' => self::createTestThread($threadAuthor)->threadID,
                'userID' => $postAuthor->userID,
                'username' => $postAuthor->username,
            ],
        ]))->executeAction()['returnValues'];
    }

    /**
     * Creates a post modificiation log entry for testing.
     *
     * @param   UserProfile $author
     * @param   UserProfile $editor
     * @param   string      $action
     * @param   array       $additionalData
     * @return  ModificationLog
     */
    public static function createTestPostModificationLog(
        UserProfile $author,
        UserProfile $editor,
        $action,
        array $additionalData = []
    ) {
        $post = self::createTestPost($author, $author);

        $originalUser = WCF::getUser();
        WCF::getSession()->changeUser($editor->getDecoratedObject(), true);

        PostModificationLogHandler::getInstance()->add($post, $action, $additionalData);

        WCF::getSession()->changeUser($originalUser, true);

        $sql = "SELECT  *
                FROM    wcf" . WCF_N . "_modification_log
                WHERE   objectTypeID = ?
                    AND objectID = ?
                    AND action = ?";
        $statement = WCF::getDB()->prepareStatement($sql);
        $statement->execute([
            ObjectTypeCache::getInstance()
                ->getObjectTypeIDByName('com.woltlab.wcf.modifiableContent', 'com.woltlab.wbb.post'),
            $post->postID,
            $action,
        ]);

        return $statement->fetchObject(ModificationLog::class);
    }

    /**
     * Creates a thread modificiation log entry for testing.
     *
     * @param   UserProfile $author
     * @param   UserProfile $editor
     * @param   string      $action
     * @param   array       $additionalData
     * @return  ModificationLog
     */
    public static function createTestThreadModificationLog(
        UserProfile $author,
        UserProfile $editor,
        $action,
        array $additionalData = []
    ) {
        $thread = self::createTestThread($author);

        $originalUser = WCF::getUser();
        WCF::getSession()->changeUser($editor->getDecoratedObject(), true);

        ThreadModificationLogHandler::getInstance()->add($thread, $action, $additionalData);

        WCF::getSession()->changeUser($originalUser, true);

        $sql = "SELECT  *
                FROM    wcf" . WCF_N . "_modification_log
                WHERE   objectTypeID = ?
                    AND objectID = ?
                    AND action = ?";
        $statement = WCF::getDB()->prepareStatement($sql);
        $statement->execute([
            ObjectTypeCache::getInstance()
                ->getObjectTypeIDByName('com.woltlab.wcf.modifiableContent', 'com.woltlab.wbb.thread'),
            $thread->threadID,
            $action,
        ]);

        return $statement->fetchObject(ModificationLog::class);
    }

    /**
     * Creates a thread for testing.
     *
     * @param   UserProfile $author
     * @return  Thread
     */
    public static function createTestThread(UserProfile $author)
    {
        $board = self::createTestBoard();

        /** @var Thread $thread */
        $thread = (new ThreadAction([], 'create', [
            'board' => $board,
            'data' => [
                'boardID' => $board->boardID,
                'languageID' => $author->languageID,
                'topic' => 'Test Thread Topic',
                'time' => TIME_NOW,
                'userID' => $author->userID,
                'username' => $author->username,
            ],
            'postData' => [
                'message' => 'Test Thread Message',
            ],
        ]))->executeAction()['returnValues'];

        return new Thread($thread->threadID);
    }
}
