<?php

namespace wbb\data\thread;

use wbb\data\board\Board;
use wcf\system\cache\builder\ICacheBuilder;

/**
 * Represents a list of the threads whose ids are fetched from a cache builder.
 *
 * @author  Matthias Schmidt
 * @copyright   2001-2019 WoltLab GmbH
 * @license WoltLab License <http://www.woltlab.com/license-agreement.html>
 * @package WoltLabSuite\Forum\Data\Thread
 * @since   5.0
 */
class CacheBuilderThreadList extends AccessibleThreadList
{
    /**
     * id of the board to which all listed threads have to belong to or `null` if no such filtering
     * should happen
     * @var int|null
     */
    public $boardID;

    /**
     * cache builder object providing cached thread ids
     * @var ICacheBuilder
     */
    public $cacheBuilder;

    /**
     * sort order passed to the cache builder
     * @var string
     */
    public $sortOrder;

    /**
     * time frame in days to select threads from
     * @var int|null
     */
    public $timeFrame;

    /**
     * Initialized a new instance of CacheBuilderThreadList.
     *
     * @param   ICacheBuilder   $cacheBuilder   cache builder object providing cached thread ids
     */
    public function __construct(ICacheBuilder $cacheBuilder)
    {
        $this->cacheBuilder = $cacheBuilder;

        parent::__construct(false);
    }

    /**
     * Returns all necessary parameters for the cache builder of the thread ids.
     *
     * @return  array
     */
    protected function getCacheBuilderParameters()
    {
        $parameters = ['limit' => $this->sqlLimit];
        if ($this->boardID) {
            $parameters['boardID'] = $this->boardID;
        }
        if ($this->sortOrder === 'ASC' || $this->sortOrder === 'DESC') {
            $parameters['sortOrder'] = $this->sortOrder;
        }
        if ($this->timeFrame) {
            $parameters['timeFrame'] = $this->timeFrame;
        }

        return $parameters;
    }

    /**
     * @inheritDoc
     */
    public function readObjectIDs()
    {
        if ($this->cacheBuilder === null) {
            parent::readObjectIDs();

            return;
        }
        $this->objectIDs = [];

        $cachedThreadIDs = $this->cacheBuilder->getData($this->getCacheBuilderParameters());
        if (empty($cachedThreadIDs)) {
            // no (relevant) threads exist, thus this list remain empty
            return;
        }

        // filter by board id
        $boardIDs = Board::filterBoardIDs(Board::getAccessibleBoardIDs());
        $privateBoardIDs = Board::filterBoardIDs(Board::getPrivateBoardIDs());
        $threadIDs = [];
        foreach ($cachedThreadIDs as $threadID => $boardID) {
            if (\in_array($boardID, $boardIDs) || \in_array($boardID, $privateBoardIDs)) {
                $threadIDs[] = $threadID;
            }
        }

        $conditionBuilderBackup = clone $this->conditionBuilder;

        // only consider cached thread
        if (!empty($threadIDs)) {
            $this->getConditionBuilder()->add('thread.threadID IN (?)', [$threadIDs]);
        } else {
            $this->getConditionBuilder()->add('1=0');
        }

        // filter cached thread ids
        parent::readObjectIDs();

        // check if the cached thread contain a sufficient number of thread accessible for the active user,
        // otherwise fallback to the pre-cached stage and read the object directly
        if (\count($this->objectIDs) < $this->sqlLimit && \count($threadIDs) >= $this->sqlLimit) {
            $this->objectIDs = [];
            $this->conditionBuilder = $conditionBuilderBackup;
            $this->applyBoardFilter();

            if ($this->boardID) {
                $this->conditionBuilder->add('thread.boardID = ?', [$this->boardID]);
            }

            if ($this->timeFrame) {
                $this->conditionBuilder->add('thread.time > ?', [TIME_NOW - $this->timeFrame * 86400]);
            }

            parent::readObjectIDs();
        }
    }

    /**
     * @inheritDoc
     */
    public function readObjects()
    {
        $this->applyBoardFilter();

        parent::readObjects();
    }
}
