<?php

namespace wbb\system\condition;

use wbb\data\board\Board;
use wbb\data\board\BoardCache;
use wbb\data\board\BoardNodeList;
use wcf\data\condition\Condition;
use wcf\system\condition\AbstractMultiSelectCondition;
use wcf\system\condition\AbstractSingleFieldCondition;
use wcf\system\condition\IObjectCondition;
use wcf\system\condition\IObjectListCondition;
use wcf\system\WCF;

/**
 * Abstract condition implementation for the board an object belongs to.
 *
 * @author  Matthias Schmidt
 * @copyright   2001-2019 WoltLab GmbH
 * @license WoltLab License <http://www.woltlab.com/license-agreement.html>
 * @package WoltLabSuite\Forum\System\Condition
 * @since   5.0
 */
abstract class AbstractObjectBoardCondition extends AbstractMultiSelectCondition implements
    IObjectCondition,
    IObjectListCondition
{
    /**
     * is `true` if the logic should be reversed, thus all of the non-selected boards fulfill the
     * condition
     * @var bool
     */
    protected $reverseLogic = false;

    /**
     * @inheritDoc
     */
    public function getData()
    {
        if (!empty($this->fieldValue)) {
            return [
                $this->fieldName => $this->fieldValue,
                $this->fieldName . '_reverseLogic' => $this->reverseLogic,
            ];
        }

        return null;
    }

    /**
     * @inheritDoc
     */
    protected function getFieldElement()
    {
        $boardNodeList = new BoardNodeList();
        $boardNodeList->readNodeTree();

        return WCF::getTPL()->fetch('__scrollableBoardCheckboxList', 'wbb', [
            'boardNodeList' => $boardNodeList->getNodeList(),
            'fieldName' => $this->fieldName,
            'fieldValue' => $this->fieldValue,
        ]);
    }

    /**
     * @inheritDoc
     */
    public function getHTML()
    {
        return WCF::getTPL()->fetch('__boardSelectionCondition', 'wbb', [
            'condition' => $this,
            'conditionHtml' => AbstractSingleFieldCondition::getHTML(),
            'fieldName' => $this->fieldName,
            'reverseLogic' => $this->reverseLogic,
        ]);
    }

    /**
     * @inheritDoc
     */
    protected function getOptions()
    {
        return \array_filter(BoardCache::getInstance()->getBoards(), static function (Board $board) {
            return !$board->isExternalLink() && (!$board->isCategory() || !empty(BoardCache::getInstance()->getChildIDs($board->boardID)));
        });
    }

    /**
     * @inheritDoc
     */
    public function readFormParameters()
    {
        parent::readFormParameters();

        if (isset($_POST[$this->fieldName . '_reverseLogic'])) {
            $this->reverseLogic = true;
        }
    }

    /**
     * @inheritDoc
     */
    public function reset()
    {
        $this->fieldValue = [];
        $this->reverseLogic = false;
    }

    /**
     * @inheritDoc
     */
    public function setData(Condition $condition)
    {
        parent::setData($condition);

        // backwards compatibility: if the reverse logic condition entry does not exist,
        // the logic is not reversed
        $this->reverseLogic = $condition->conditionData[$this->fieldName . '_reverseLogic'] ?? false;
    }
}
