<?php

namespace wbb\system\condition\post;

use wbb\data\post\Post;
use wbb\data\post\PostList;
use wcf\data\condition\Condition;
use wcf\data\DatabaseObject;
use wcf\data\DatabaseObjectDecorator;
use wcf\data\DatabaseObjectList;
use wcf\system\condition\AbstractSingleFieldCondition;
use wcf\system\condition\IObjectListCondition;
use wcf\system\exception\InvalidObjectArgument;
use wcf\system\exception\UserInputException;
use wcf\system\WCF;

/**
 * Condition implementation for the state of a post.
 *
 * @author  Matthias Schmidt
 * @copyright   2001-2019 WoltLab GmbH
 * @license WoltLab License <http://www.woltlab.com/license-agreement.html>
 * @package WoltLabSuite\Forum\System\Condition\Post
 * @since   5.0
 */
class PostStateCondition extends AbstractSingleFieldCondition implements IObjectListCondition
{
    /**
     * @inheritDoc
     */
    protected $label = 'wbb.post.condition.state';

    /**
     * values of the possible state
     * @var int[]
     */
    public $states = [
        'isDeleted' => 0,
        'isNotDeleted' => 0,
        'isDisabled' => 0,
        'isEnabled' => 0,
        'isClosed' => 0,
        'isOpen' => 0,
    ];

    /**
     * @inheritDoc
     */
    public function addObjectListCondition(DatabaseObjectList $objectList, array $conditionData)
    {
        if (!($objectList instanceof PostList)) {
            throw new InvalidObjectArgument($objectList, PostList::class, 'Object list');
        }

        if (isset($conditionData['isDeleted'])) {
            $objectList->getConditionBuilder()->add('post.isDeleted = ?', [$conditionData['isDeleted']]);
        }

        if (isset($conditionData['isDisabled'])) {
            $objectList->getConditionBuilder()->add('post.isDisabled = ?', [$conditionData['isDisabled']]);
        }

        if (isset($conditionData['isClosed'])) {
            $objectList->getConditionBuilder()->add('post.isClosed = ?', [$conditionData['isClosed']]);
        }
    }

    /**
     * @inheritDoc
     */
    public function checkObject(DatabaseObject $object, array $conditionData)
    {
        if (
            !($object instanceof Post)
            && (!($object instanceof DatabaseObjectDecorator) || !($object->getDecoratedObject() instanceof Post))
        ) {
            throw new InvalidObjectArgument($object, Post::class);
        }

        $simpleStates = ['isDeleted', 'isDisabled', 'isClosed'];
        foreach ($simpleStates as $state) {
            /** @noinspection PhpVariableVariableInspection */
            if (isset($conditionData[$state]) && $object->{$state} != $conditionData[$state]) {
                return false;
            }
        }

        return true;
    }

    /**
     * @inheritDoc
     */
    public function getData()
    {
        $data = [];

        if ($this->states['isDeleted']) {
            $data['isDeleted'] = 1;
        } elseif ($this->states['isNotDeleted']) {
            $data['isDeleted'] = 0;
        }

        if ($this->states['isDisabled']) {
            $data['isDisabled'] = 1;
        } elseif ($this->states['isEnabled']) {
            $data['isDisabled'] = 0;
        }

        if ($this->states['isClosed']) {
            $data['isClosed'] = 1;
        } elseif ($this->states['isOpen']) {
            $data['isClosed'] = 0;
        }

        if (!empty($data)) {
            return $data;
        }

        return null;
    }

    /**
     * @inheritDoc
     */
    protected function getFieldElement()
    {
        $fieldElement = '';

        foreach ($this->states as $state => $value) {
            $fieldElement .= '<label><input type="checkbox" name="wbbPost' . \ucfirst($state) . '" value="1"' . ($value ? ' checked' : '') . '> ' . WCF::getLanguage()->get('wbb.post.condition.state.' . $state) . "</label>\n";
        }

        return $fieldElement;
    }

    /**
     * @inheritDoc
     */
    public function readFormParameters()
    {
        foreach ($this->states as $state => &$value) {
            if (isset($_POST['wbbPost' . \ucfirst($state)])) {
                $value = 1;
            }
        }
        unset($value);
    }

    /**
     * @inheritDoc
     */
    public function reset()
    {
        foreach ($this->states as $state => $value) {
            $this->states[$state] = 0;
        }
    }

    /**
     * @inheritDoc
     */
    public function setData(Condition $condition)
    {
        /** @noinspection PhpUndefinedFieldInspection */
        $isDeleted = $condition->isDeleted;
        if ($isDeleted !== null) {
            $this->states['isDeleted'] = $isDeleted;
            $this->states['isNotDeleted'] = 1 - $isDeleted;
        }

        /** @noinspection PhpUndefinedFieldInspection */
        $isDisabled = $condition->isDisabled;
        if ($isDisabled !== null) {
            $this->states['isDisabled'] = $isDisabled;
            $this->states['isEnabled'] = 1 - $isDisabled;
        }

        /** @noinspection PhpUndefinedFieldInspection */
        $isClosed = $condition->isClosed;
        if ($isClosed !== null) {
            $this->states['isClosed'] = $isClosed;
            $this->states['isOpen'] = 1 - $isClosed;
        }
    }

    /**
     * @inheritDoc
     */
    public function validate()
    {
        if ($this->states['isDeleted'] && $this->states['isNotDeleted']) {
            $this->errorMessage = 'wbb.post.condition.state.isDeleted.error.conflict';

            throw new UserInputException('isDeleted', 'conflict');
        }

        if ($this->states['isDisabled'] && $this->states['isEnabled']) {
            $this->errorMessage = 'wbb.post.condition.state.isDisabled.error.conflict';

            throw new UserInputException('isDisabled', 'conflict');
        }

        if ($this->states['isClosed'] && $this->states['isOpen']) {
            $this->errorMessage = 'wbb.post.condition.state.isClosed.error.conflict';

            throw new UserInputException('isClosed', 'conflict');
        }
    }
}
