<?php

namespace wbb\data\post;

use wbb\data\thread\Thread;
use wbb\data\thread\ThreadList;
use wbb\system\option\ThreadFormOptionHandler;
use wcf\data\DatabaseObjectEditor;
use wcf\data\TFastCreate;
use wcf\system\event\EventHandler;
use wcf\system\language\LanguageFactory;
use wcf\system\search\SearchIndexManager;
use wcf\system\WCF;

/**
 * Provides functions to edit posts.
 *
 * @author  Marcel Werk
 * @copyright   2001-2019 WoltLab GmbH
 * @license WoltLab License <http://www.woltlab.com/license-agreement.html>
 * @package WoltLabSuite\Forum\Data\Post
 *
 * @method static   Post    create(array $parameters = [])
 * @method      Post    getDecoratedObject()
 * @mixin       Post
 */
class PostEditor extends DatabaseObjectEditor
{
    use TFastCreate;

    /**
     * @inheritDoc
     */
    protected static $baseClass = Post::class;

    /**
     * Updates the post counter of the given users.
     *
     * @param   array       $users      user id => post counter increase/decrease
     */
    public static function updatePostCounter(array $users)
    {
        $sql = "UPDATE  wcf" . WCF_N . "_user
                SET     wbbPosts = wbbPosts + ?
                WHERE   userID = ?";
        $statement = WCF::getDB()->prepareStatement($sql);
        foreach ($users as $userID => $posts) {
            $statement->execute([$posts, $userID]);
        }
    }

    /**
     * Adds the posts with the given postIDs to the search index.
     *
     * @param       int[]               $postIDs
     * @since       5.2
     */
    public static function addPostIDsToSearchIndex(array $postIDs)
    {
        $postList = new PostList();
        $postList->setObjectIDs($postIDs);
        $postList->readObjects();

        self::addToSearchIndex($postList->getObjects());
    }

    /**
     * Adds posts to the search index.
     *
     * @param       Post[]                  $posts
     * @param       bool                 $fetchThread
     * @since       5.2
     */
    public static function addToSearchIndex(array $posts, $fetchThread = true)
    {
        $parameters = [
            'posts' => $posts,
        ];
        EventHandler::getInstance()->fireAction(self::class, 'addToSearchIndex', $parameters);
        $posts = $parameters['posts'];
        if ($posts === []) {
            return;
        }

        if ($fetchThread) {
            $threadIDs = [];
            foreach ($posts as $post) {
                if (!($post instanceof Post) && !($post instanceof self)) {
                    throw new \InvalidArgumentException("The object '" . \get_class($post) . "' is not an instance of the class '" . Post::class . "'");
                }

                $threadIDs[] = $post->threadID;
            }

            $threadList = new ThreadList();
            $threadList->setObjectIDs($threadIDs);
            $threadList->readObjects();
            $threads = $threadList->getObjects();
        }

        foreach ($posts as $post) {
            if ($fetchThread) {
                if (!isset($threads[$post->threadID])) {
                    throw new \LogicException("Unable to find " . Thread::class . " with the threadID '" . $post->threadID . "'.");
                }

                $post->setThread($threads[$post->threadID]);
            }

            if ($post instanceof self) {
                $post = $post->getDecoratedObject();
            }

            $metaData = "";

            if ($post->isFirstPost()) {
                $optionHandler = new ThreadFormOptionHandler(false);
                $optionHandler->setPost($post);

                $language = LanguageFactory::getInstance()->getLanguage(
                    $post->getThread()->languageID ?: LanguageFactory::getInstance()->getDefaultLanguageID()
                );

                foreach ($optionHandler->getOptions() as $option) {
                    if ($option['object']->getOptionValue()) {
                        if (!empty($metaData)) {
                            $metaData .= "\n";
                        }
                        $metaData .= $language->get($option['object']->optionTitle) . ": " . $option['object']->getFormattedOptionValue();
                    }
                }
            }

            // update search index
            SearchIndexManager::getInstance()->set(
                'com.woltlab.wbb.post',
                $post->postID,
                $post->message,
                $post->subject,
                $post->time,
                $post->userID,
                $post->username,
                $post->getThread()->languageID,
                $metaData
            );
        }
    }
}
