<?php

namespace wbb\system\bulk\processing\thread;

use wbb\data\board\BoardCache;
use wbb\data\board\BoardNodeList;
use wbb\data\thread\ThreadAction;
use wbb\data\thread\ThreadList;
use wcf\data\DatabaseObjectList;
use wcf\system\exception\InvalidObjectArgument;
use wcf\system\exception\UserInputException;
use wcf\system\WCF;

/**
 * Bulk processing action implementation for moving threads to a different board.
 *
 * @author  Matthias Schmidt
 * @copyright   2001-2019 WoltLab GmbH
 * @license WoltLab License <http://www.woltlab.com/license-agreement.html>
 * @package WoltLabSuite\Forum\System\Bulk\Processing\Thread
 * @since   5.0
 */
class MoveThreadBulkProcessingAction extends AbstractThreadBulkProcessingAction
{
    /**
     * id of the target board
     * @var int
     */
    public $targetBoardID = 0;

    /**
     * @inheritDoc
     */
    public function executeAction(DatabaseObjectList $objectList)
    {
        if (!($objectList instanceof ThreadList)) {
            throw new InvalidObjectArgument($objectList, ThreadList::class, 'Object list');
        }

        /** @noinspection PhpUndefinedFieldInspection */
        if ($this->object->action === null) {
            return;
        }

        if (\count($objectList)) {
            $threadAction = new ThreadAction($objectList->getObjects(), 'move', [
                'boardID' => $this->targetBoardID,
                'isBulkProcessing' => true,
                'showMoveNotice' => false,
            ]);
            $threadAction->executeAction();
        }
    }

    /**
     * @inheritDoc
     */
    public function getHTML()
    {
        $boardNodeList = new BoardNodeList();
        $boardNodeList->readNodeTree();

        return WCF::getTPL()->fetch('moveThreadBulkProcessing', 'wbb', [
            'boardNodeList' => $boardNodeList->getNodeList(),
            'targetBoardID' => $this->targetBoardID,
        ]);
    }

    /**
     * @inheritDoc
     */
    public function readFormParameters()
    {
        if (isset($_POST['targetBoardID'])) {
            $this->targetBoardID = \intval($_POST['targetBoardID']);
        }
    }

    /**
     * @inheritDoc
     */
    public function reset()
    {
        $this->targetBoardID = 0;
    }

    /**
     * @inheritDoc
     */
    public function validate()
    {
        if (!$this->targetBoardID) {
            throw new UserInputException('targetBoardID');
        }

        $board = BoardCache::getInstance()->getBoard($this->targetBoardID);
        if ($board === null || !$board->isBoard()) {
            throw new UserInputException('targetBoardID', 'noValidSelection');
        }
    }
}
