<?php

namespace wbb\data\thread;

use wbb\data\post\Post;
use wcf\data\IFeedEntry;
use wcf\system\request\LinkHandler;
use wcf\system\WCF;
use wcf\util\StringUtil;

/**
 * Represents a viewable thread for RSS feeds.
 *
 * @author  Marcel Werk
 * @copyright   2001-2019 WoltLab GmbH
 * @license WoltLab License <http://www.woltlab.com/license-agreement.html>
 * @package WoltLabSuite\Forum\Data\Thread
 *
 * @property    string      $message        text of the thread's last post
 */
class FeedThread extends ViewableThread implements IFeedEntry
{
    /**
     * @var Post
     */
    protected $post;

    /**
     * @inheritDoc
     */
    public function getLink()
    {
        return LinkHandler::getInstance()->getLink('Thread', [
            'application' => 'wbb',
            'object' => $this->getDecoratedObject(),
            'encodeTitle' => true,
        ]);
    }

    /**
     * @inheritDoc
     */
    public function getTitle()
    {
        return $this->getDecoratedObject()->getTitle();
    }

    /**
     * @inheritDoc
     */
    public function getFormattedMessage()
    {
        return $this->getPost()->getFormattedMessage();
    }

    /**
     * Returns a simplified version of the formatted message.
     *
     * @return  string
     */
    public function getSimplifiedFormattedMessage()
    {
        return $this->getPost()->getSimplifiedFormattedMessage();
    }

    /**
     * @inheritDoc
     */
    public function getMessage()
    {
        return $this->message;
    }

    /**
     * @inheritDoc
     */
    public function getExcerpt($maxLength = 255)
    {
        return StringUtil::truncateHTML($this->getSimplifiedFormattedMessage(), $maxLength);
    }

    /**
     * @inheritDoc
     */
    public function getUserID()
    {
        return $this->lastPosterID;
    }

    /**
     * @inheritDoc
     */
    public function getUsername()
    {
        return $this->lastPoster;
    }

    /**
     * @inheritDoc
     */
    public function getTime()
    {
        return $this->lastPostTime;
    }

    /**
     * @inheritDoc
     */
    public function __toString()
    {
        return $this->getFormattedMessage();
    }

    /**
     * @inheritDoc
     */
    public function getComments()
    {
        return $this->replies;
    }

    /**
     * @inheritDoc
     */
    public function getCategories()
    {
        return [];
    }

    /**
     * @inheritDoc
     */
    public function isVisible()
    {
        return $this->canRead();
    }

    /**
     * @return Post
     */
    protected function getPost()
    {
        if ($this->post === null) {
            $message = $this->message;
            if (WBB_THREAD_GUEST_MAX_POSTS && !WCF::getUser()->userID && $this->replies >= WBB_THREAD_GUEST_MAX_POSTS) {
                $message = '';
            }

            $this->post = new Post(null, [
                'postID' => $this->lastPostID,
                'message' => $message,
            ]);
        }

        return $this->post;
    }
}
