<?php

namespace wbb\system\thread\editor;

use wbb\data\board\BoardCache;
use wbb\data\thread\ThreadEditor;
use wbb\system\label\object\ThreadLabelObjectHandler;
use wbb\system\log\modification\ThreadModificationLogHandler;
use wcf\data\label\LabelList;
use wcf\system\exception\UserInputException;
use wcf\system\label\LabelHandler;
use wcf\system\request\LinkHandler;
use wcf\system\WCF;

/**
 * Provides an editor for thread labels.
 *
 * @author  Alexander Ebert
 * @copyright   2001-2019 WoltLab GmbH
 * @license WoltLab License <http://www.woltlab.com/license-agreement.html>
 * @package WoltLabSuite\Forum\System\Thread\Editor
 */
class LabelThreadEditor extends AbstractThreadEditor
{
    /**
     * list of label ids
     * @var int[]
     */
    public $labelIDs = [];

    /**
     * @inheritDoc
     */
    public function beginEdit()
    {
        $labelGroupIDs = BoardCache::getInstance()->getLabelGroupIDs($this->thread->boardID);
        if (empty($labelGroupIDs)) {
            // nothing to do here :)
            return '';
        }

        $assignedLabels = ThreadLabelObjectHandler::getInstance()->getAssignedLabels([$this->thread->threadID]);
        if (isset($assignedLabels[$this->thread->threadID])) {
            $assignedLabels = $assignedLabels[$this->thread->threadID];
        }

        WCF::getTPL()->assign([
            'assignedLabels' => $assignedLabels,
            'labelGroups' => LabelHandler::getInstance()->getLabelGroups($labelGroupIDs),
        ]);

        return WCF::getTPL()->fetch('threadEditorLabel', 'wbb');
    }

    /**
     * @inheritDoc
     */
    public function validate(array &$data)
    {
        // get already assigned label ids
        $labelIDs = ThreadLabelObjectHandler::getInstance()->getAssignedLabels([$this->thread->threadID]);
        if (isset($labelIDs[$this->thread->threadID])) {
            foreach ($labelIDs[$this->thread->threadID] as $label) {
                $this->labelIDs[$label->groupID] = $label->labelID;
            }
        }

        $labelGroupIDs = BoardCache::getInstance()->getLabelGroupIDs($this->thread->boardID);
        $labelGroups = LabelHandler::getInstance()->getLabelGroups($labelGroupIDs);
        if (!empty($labelGroupIDs)) {
            foreach ($data as $groupID => $labelID) {
                if (\is_array($labelID) || !$labelID) {
                    unset($this->labelIDs[$groupID]);
                    continue;
                }

                if (!isset($labelGroups[$groupID]) || !$labelGroups[$groupID]->isValid($labelID)) {
                    unset($this->labelIDs[$groupID]);
                    continue;
                }

                $this->labelIDs[$groupID] = $labelID;
            }
        }

        // check if required label groups were set
        $optionID = LabelHandler::getInstance()->getOptionID('canSetLabel');
        $validationErrors = [];
        foreach ($labelGroups as $labelGroup) {
            if ($labelGroup->forceSelection && !isset($this->labelIDs[$labelGroup->groupID])) {
                // check if group wasn't set, but is not accessible for this user anyway
                if (!$labelGroup->getPermission($optionID)) {
                    continue;
                }

                $validationErrors[$labelGroup->groupID] = 'missing';
            }
        }

        if (!empty($validationErrors)) {
            throw new UserInputException('labelIDs', $validationErrors);
        }
    }

    /**
     * @inheritDoc
     */
    public function saveEdit(array $data)
    {
        // get label data
        $labelList = null;
        if (!empty($this->labelIDs)) {
            $labelList = new LabelList();
            $labelList->setObjectIDs($this->labelIDs);
            $labelList->readObjects();
        }

        // get old labels
        $oldLabels = [];
        $assignedLabels = ThreadLabelObjectHandler::getInstance()->getAssignedLabels([$this->thread->threadID]);
        if (isset($assignedLabels[$this->thread->threadID])) {
            foreach ($assignedLabels[$this->thread->threadID] as $label) {
                $oldLabels[$label->groupID] = $label;
            }
        }

        // log changes
        if ($labelList !== null) {
            foreach ($labelList as $label) {
                if (!isset($oldLabels[$label->groupID]) || $label->labelID != $oldLabels[$label->groupID]->labelID) {
                    ThreadModificationLogHandler::getInstance()->setLabel(
                        $this->thread,
                        $label,
                        ($oldLabels[$label->groupID] ?? null)
                    );
                }
                if (isset($oldLabels[$label->groupID])) {
                    unset($oldLabels[$label->groupID]);
                }
            }
        }
        foreach ($oldLabels as $label) {
            ThreadModificationLogHandler::getInstance()->setLabel($this->thread, null, $label);
        }

        // save changes
        ThreadLabelObjectHandler::getInstance()->setLabels($this->labelIDs, $this->thread->threadID);
        $assignedLabels = ThreadLabelObjectHandler::getInstance()->getAssignedLabels([$this->thread->threadID], false);
        $editor = new ThreadEditor($this->thread);
        $editor->update([
            'hasLabels' => !empty($assignedLabels[$this->thread->threadID]) ? 1 : 0,
        ]);

        $labels = [];
        if ($labelList !== null) {
            $tmp = [];
            foreach ($labelList as $label) {
                $tmp[$label->labelID] = [
                    'cssClassName' => $label->cssClassName,
                    'label' => $label->getTitle(),
                    'link' => LinkHandler::getInstance()->getLink(
                        'Board',
                        ['application' => 'wbb', 'object' => $this->thread->getBoard()],
                        'labelIDs[' . $label->groupID . ']=' . $label->labelID
                    ),
                ];
            }

            // sort labels by label group show order
            $labelGroups = ThreadLabelObjectHandler::getInstance()->getLabelGroups();
            foreach ($labelGroups as $labelGroup) {
                foreach ($tmp as $labelID => $labelData) {
                    if ($labelGroup->isValid($labelID)) {
                        $labels[] = $labelData;
                        break;
                    }
                }
            }
        }

        return [
            'labels' => $labels,
        ];
    }
}
