/**
 * Handles toggling watched boards in the list of all watched boards.
 *
 * @author  Matthias Schmidt
 * @copyright  2001-2019 WoltLab GmbH
 * @license  WoltLab License <http://www.woltlab.com/license-agreement.html>
 * @module  WoltLabSuite/Forum/Controller/Board/WatchedList
 */
define(["require", "exports", "tslib", "WoltLabSuite/Core/Ajax", "WoltLabSuite/Core/Core", "WoltLabSuite/Core/Language", "WoltLabSuite/Core/Ui/Confirmation"], function (require, exports, tslib_1, Ajax, Core, Language, UiConfirmation) {
    "use strict";
    Object.defineProperty(exports, "__esModule", { value: true });
    exports.setup = void 0;
    Ajax = tslib_1.__importStar(Ajax);
    Core = tslib_1.__importStar(Core);
    Language = tslib_1.__importStar(Language);
    UiConfirmation = tslib_1.__importStar(UiConfirmation);
    let _button;
    const _checkboxes = [];
    let _markAllButton;
    /**
     * Returns the ids of the selected watched boards.
     */
    function getSelectedBoardIDs() {
        return _checkboxes.filter((checkbox) => checkbox.checked).map((checkbox) => ~~checkbox.dataset.objectId);
    }
    /**
     * Handles changing a mark checkbox.
     */
    function mark() {
        updateActionButtonLabel();
        updateMarkAllButton();
    }
    /**
     * Handles clicking on the 'mark all' button.
     */
    function markAll() {
        const isChecked = Core.stringToBool(_markAllButton.dataset.hasUnmarkedBoards);
        _checkboxes.forEach((checkbox) => (checkbox.checked = isChecked));
        updateActionButtonLabel();
        updateMarkAllButton();
    }
    /**
     * Handles a click on the stop watching button.
     */
    function stopWatching() {
        const boardIds = getSelectedBoardIDs();
        const type = boardIds.length ? "stopWatchingMarked" : "stopWatchingAll";
        const languageItem = `wbb.board.watchedBoards.${type}.confirmMessage`;
        UiConfirmation.show({
            confirm() {
                Ajax.apiOnce({
                    data: {
                        actionName: "stopWatching",
                        className: "wbb\\data\\board\\BoardAction",
                        objectIDs: boardIds,
                    },
                    success() {
                        window.location.reload();
                    },
                });
            },
            message: Language.get(languageItem),
        });
    }
    /**
     * Updates the label of the action button.
     */
    function updateActionButtonLabel() {
        const count = getSelectedBoardIDs().length;
        if (count) {
            _button.textContent = Language.get("wbb.board.watchedBoards.stopWatchingMarked", {
                count: count,
            });
        }
        else {
            _button.textContent = Language.get("wbb.board.watchedBoards.stopWatchingAll");
        }
    }
    /**
     * Updates the label of mark all button.
     */
    function updateMarkAllButton() {
        const hasUnmarkedBoards = _checkboxes.some((checkbox) => !checkbox.checked);
        _markAllButton.dataset.hasUnmarkedBoards = hasUnmarkedBoards ? "true" : "false";
        const type = hasUnmarkedBoards ? "markAll" : "unmarkAll";
        _markAllButton.textContent = Language.get(`wbb.board.watchedBoards.${type}`);
    }
    /**
     * Initializes the watched board list.
     */
    function setup() {
        document.querySelectorAll(".wbbBoard .containerHeadline > h3").forEach((boardTitle) => {
            const checkbox = document.createElement("input");
            checkbox.type = "checkbox";
            const boardContainer = boardTitle.closest(".wbbBoardContainer");
            checkbox.dataset.objectId = boardContainer.dataset.boardId;
            checkbox.addEventListener("change", mark);
            checkbox.style.setProperty("margin-right", "7px", "");
            _checkboxes.push(checkbox);
            boardTitle.insertAdjacentElement("afterbegin", checkbox);
        });
        _button = document.getElementById("stopWatchingButton");
        _button.addEventListener("click", stopWatching);
        _markAllButton = document.getElementById("markAllWatchedBoards");
        _markAllButton.addEventListener("click", markAll);
        _markAllButton.dataset.hasUnmarkedBoards = "1";
    }
    exports.setup = setup;
});
