<?php

namespace wbb\data\thread;

use wbb\system\cache\builder\LatestPostsCacheBuilder;
use wcf\system\database\util\PreparedStatementConditionBuilder;
use wcf\system\language\LanguageFactory;
use wcf\system\WCF;

/**
 * Represents a list of the latest posts.
 *
 * @author  Marcel Werk
 * @copyright   2001-2019 WoltLab GmbH
 * @license WoltLab License <http://www.woltlab.com/license-agreement.html>
 * @package WoltLabSuite\Forum\Data\Thread
 */
class LatestPostsList extends ViewableThreadList
{
    /**
     * @inheritDoc
     */
    public $sqlLimit = 5;

    /**
     * @inheritDoc
     */
    public $sqlOrderBy = 'thread.lastPostTime DESC';

    /**
     * Creates a new LatestPostsList object.
     */
    public function __construct()
    {
        parent::__construct();
        $this->objectIDs = [];

        // get cache
        $cachedThreadIDs = \array_keys(LatestPostsCacheBuilder::getInstance()->getData());
        if (!\count($cachedThreadIDs)) {
            return;
        }

        // filter by board id
        if (!$this->applyBoardFilter(true)) {
            return;
        }

        // add conditions
        $this->getConditionBuilder()->add('thread.threadID IN (?)', [$cachedThreadIDs]);
        $this->getConditionBuilder()->add(
            'thread.isDeleted = 0 AND thread.isDisabled = 0 AND thread.movedThreadID IS NULL'
        );
        // apply language filter
        if (LanguageFactory::getInstance()->multilingualismEnabled() && \count(WCF::getUser()->getLanguageIDs())) {
            $this->getConditionBuilder()->add(
                '(thread.languageID IN (?) OR thread.languageID IS NULL)',
                [WCF::getUser()->getLanguageIDs()]
            );
        }

        // filter cache threads
        $this->readObjectIDs();

        // cached threads didn't contain the latest X threads for the active user
        // fallback to non-cached function
        if (\count($this->objectIDs) < $this->sqlLimit && \count($cachedThreadIDs) >= $this->sqlLimit) {
            $this->objectIDs = [];
            $this->conditionBuilder = new PreparedStatementConditionBuilder();
            $this->applyBoardFilter(true);
            $this->getConditionBuilder()->add(
                'thread.isDeleted = 0 AND thread.isDisabled = 0 AND thread.movedThreadID IS NULL'
            );
            // apply language filter
            if (LanguageFactory::getInstance()->multilingualismEnabled() && \count(WCF::getUser()->getLanguageIDs())) {
                $this->getConditionBuilder()->add(
                    '(thread.languageID IN (?) OR thread.languageID IS NULL)',
                    [WCF::getUser()->getLanguageIDs()]
                );
            }

            $this->readObjectIDs();
        }
    }
}
