<?php

namespace wbb\system\attachment;

use wbb\data\board\BoardCache;
use wbb\data\post\Post;
use wbb\data\post\PostList;
use wbb\data\thread\Thread;
use wbb\system\cache\runtime\PostRuntimeCache;
use wcf\system\attachment\AbstractAttachmentObjectType;
use wcf\system\WCF;

/**
 * Attachment object type implementation for posts.
 *
 * @author  Marcel Werk
 * @copyright   2001-2019 WoltLab GmbH
 * @license WoltLab License <http://www.woltlab.com/license-agreement.html>
 * @package WoltLabSuite\Forum\System\Attachment
 *
 * @method  Post    getObject($objectID)
 */
class PostAttachmentObjectType extends AbstractAttachmentObjectType
{
    /**
     * @inheritDoc
     */
    public function canDownload($objectID)
    {
        if ($objectID) {
            $post = PostRuntimeCache::getInstance()->getObject($objectID);
            if (!$post || !$post->canRead()) {
                return false;
            }

            $board = BoardCache::getInstance()->getBoard($post->getThread()->boardID);

            return $board->getPermission('canDownloadAttachment');
        }

        return false;
    }

    /**
     * @inheritDoc
     */
    public function canViewPreview($objectID)
    {
        if ($objectID) {
            $post = PostRuntimeCache::getInstance()->getObject($objectID);
            if (!$post || !$post->canRead()) {
                return false;
            }

            $board = BoardCache::getInstance()->getBoard($post->getThread()->boardID);

            return $board->getPermission('canViewAttachmentPreview');
        }

        return false;
    }

    /**
     * @inheritDoc
     */
    public function canUpload($objectID, $parentObjectID = 0)
    {
        $board = null;
        if ($objectID) {
            $post = PostRuntimeCache::getInstance()->getObject($objectID);
            if (!$post || !$post->getThread()->canEditPost($post)) {
                return false;
            }

            $board = BoardCache::getInstance()->getBoard($post->getThread()->boardID);
        } elseif ($parentObjectID) {
            $board = BoardCache::getInstance()->getBoard($parentObjectID);
        }

        if ($board !== null) {
            return $board->getPermission('canUploadAttachment');
        }

        return false;
    }

    /**
     * @inheritDoc
     */
    public function canDelete($objectID)
    {
        if ($objectID) {
            $post = PostRuntimeCache::getInstance()->getObject($objectID);
            if ($post && $post->getThread()->canEditPost($post)) {
                return true;
            }
        }

        return false;
    }

    /**
     * @inheritDoc
     */
    public function cacheObjects(array $objectIDs)
    {
        $postList = new PostList();
        $postList->sqlSelects .= 'thread.boardID, thread.topic';
        $postList->sqlJoins .= '
            LEFT JOIN   wbb' . WCF_N . '_thread thread
            ON          thread.threadID = post.threadID';
        $postList->setObjectIDs($objectIDs);
        $postList->readObjects();
        foreach ($postList as $post) {
            /** @noinspection PhpUndefinedFieldInspection */
            $post->setThread(new Thread(null, [
                'threadID' => $post->threadID,
                'boardID' => $post->boardID,
                'topic' => $post->topic,
            ]));
        }

        foreach ($postList->getObjects() as $objectID => $object) {
            $this->cachedObjects[$objectID] = $object;
        }
    }

    /**
     * @inheritDoc
     */
    public function setPermissions(array $attachments)
    {
        $postIDs = [];
        foreach ($attachments as $attachment) {
            // set default permissions
            $attachment->setPermissions([
                'canDownload' => false,
                'canViewPreview' => false,
            ]);

            if ($this->getObject($attachment->objectID) === null) {
                $postIDs[] = $attachment->objectID;
            }
        }

        if (!empty($postIDs)) {
            $this->cacheObjects($postIDs);
        }

        foreach ($attachments as $attachment) {
            if (($post = $this->getObject($attachment->objectID)) !== null) {
                if (!$post->canRead()) {
                    continue;
                }
                $board = BoardCache::getInstance()->getBoard($post->getThread()->boardID);

                $attachment->setPermissions([
                    'canDownload' => $board->getPermission('canDownloadAttachment'),
                    'canViewPreview' => $board->getPermission('canViewAttachmentPreview'),
                ]);
            } elseif ($attachment->tmpHash != '' && $attachment->userID == WCF::getUser()->userID) {
                $attachment->setPermissions([
                    'canDownload' => true,
                    'canViewPreview' => true,
                ]);
            }
        }
    }
}
