<?php

namespace wbb\system\board\status;

use InvalidArgumentException;
use wbb\data\board\BoardCache;
use wbb\data\thread\Thread;
use wbb\system\thread\status\ThreadStatusHandler;
use wbb\system\user\notification\object\ThreadUserNotificationObject;
use wcf\data\user\User;
use wcf\system\database\util\PreparedStatementConditionBuilder;
use wcf\system\SingletonFactory;
use wcf\system\user\notification\UserNotificationHandler;
use wcf\system\user\storage\UserStorageHandler;
use wcf\system\WCF;

/**
 * Handles board status related actions.
 *
 * @author  Joshua Ruesweg
 * @copyright   2001-2021 WoltLab GmbH
 * @license WoltLab License <http://www.woltlab.com/license-agreement.html>
 * @package WoltLabSuite\Forum\System\Board\Status
 * @since 5.5
 */
final class BoardStatusHandler extends SingletonFactory
{
    public const SUBSCRIPTION_MODE_NORMAL = 'normal';

    public const SUBSCRIPTION_MODE_THREADSONLY = 'threadsOnly';

    public const SUBSCRIPTION_MODE_IGNORING = 'ignoring';

    public const SUBSCRIPTION_MODE_THREADSANDPOSTS = 'threadsAndPosts';

    private $statusCache = [];

    private $statusBoardCache = [];

    /**
     * @inheritDoc
     */
    protected function init()
    {
        if (WCF::getUser()->userID) {
            $this->loadCache(WCF::getUser()->userID);
        }
    }

    private function loadCache(int $userID): void
    {
        $sql = "SELECT      boardID, status
                FROM        wbb1_board_user_status
                WHERE       userID = ?";
        $statement = WCF::getDB()->prepare($sql);
        $statement->execute([
            $userID,
        ]);

        foreach (BoardCache::getInstance()->getAllChildIDs() as $boardID) {
            $this->statusCache[$userID][$boardID] = self::SUBSCRIPTION_MODE_NORMAL;
        }

        while ($row = $statement->fetchArray()) {
            $this->statusCache[$userID][$row['boardID']] = $row['status'];
        }
    }

    /**
     * Returns the subscription status for a board and a given user. If the `$user` is `null`,
     * the current user will be used.
     *
     * @throws InvalidArgumentException If the given user is a guest.
     */
    public function getSubscriptionStatus(int $boardID, ?User $user = null): string
    {
        if ($user === null) {
            $user = WCF::getUser();
        }

        if (!$user->userID) {
            throw new \InvalidArgumentException("Guests do not have any status for boards.");
        }

        if (!isset($this->statusCache[$user->userID][$boardID])) {
            $sql = "SELECT      status
                    FROM        wbb1_board_user_status
                    WHERE       userID = ?
                            AND boardID = ?";
            $statement = WCF::getDB()->prepare($sql);
            $statement->execute([
                $user->userID,
                $boardID,
            ]);

            $this->statusCache[$user->userID][$boardID] = $statement->fetchSingleColumn() ?: self::SUBSCRIPTION_MODE_NORMAL;
        }

        return $this->statusCache[$user->userID][$boardID];
    }

    /**
     * Saves the given subscription status for a board and a given user. If the `$user` is `null`,
     * the current user will be used.
     *
     * @throws InvalidArgumentException If the given user is a guest.
     */
    public function saveSubscriptionStatus(array $boardIDs, string $status, ?User $user = null): void
    {
        if ($user === null) {
            $user = WCF::getUser();
        }

        if (!$user->userID) {
            throw new \InvalidArgumentException("Guests do not have any status for boards.");
        }

        switch ($status) {
            case self::SUBSCRIPTION_MODE_NORMAL:
                $conditionBuilder = new PreparedStatementConditionBuilder();
                $conditionBuilder->add("boardID IN (?)", [$boardIDs]);
                $conditionBuilder->add("userID = ?", [$user->userID]);
                $sql = "DELETE FROM     wbb1_board_user_status
                        {$conditionBuilder}";
                $statement = WCF::getDB()->prepare($sql);
                $statement->execute($conditionBuilder->getParameters());
                break;

            case self::SUBSCRIPTION_MODE_IGNORING:
            case self::SUBSCRIPTION_MODE_THREADSONLY:
            case self::SUBSCRIPTION_MODE_THREADSANDPOSTS:
                $sql = "INSERT INTO     wbb1_board_user_status
                                        (boardID, userID, status)
                        VALUES          (?, ?, ?)
                        ON DUPLICATE KEY UPDATE status = VALUES(status)";
                $statement = WCF::getDB()->prepare($sql);
                foreach ($boardIDs as $boardID) {
                    $statement->execute([
                        $boardID,
                        $user->userID,
                        $status,
                    ]);
                }
                break;

            default:
                throw new \InvalidArgumentException(
                    \sprintf("Unknown board status '%s' given.", $status)
                );
        }

        UserStorageHandler::getInstance()->reset([$user->userID], 'wbbUnreadWatchedBoards');
        UserStorageHandler::getInstance()->reset([$user->userID], 'wbbWatchedBoards');

        foreach ($boardIDs as $boardID) {
            $this->statusCache[$user->userID][$boardID] = $status;
        }
    }

    /**
     * Deletes all subscriptions with the given status ('threadsOnly', 'ignoring' or 'threadsAndPosts').
     *
     * @throws InvalidArgumentException If the status is not 'threadsOnly', 'threadsAndPosts' nor 'ignoring'.
     */
    public function deleteAllSubscriptions(array $boardIDs, string $status): void
    {
        if (
            !\in_array($status, [
                self::SUBSCRIPTION_MODE_THREADSONLY,
                self::SUBSCRIPTION_MODE_IGNORING,
                self::SUBSCRIPTION_MODE_THREADSANDPOSTS,
            ])
        ) {
            throw new \InvalidArgumentException(\sprintf(
                "Unknown board status '%s' given.",
                $status
            ));
        }

        $conditionBuilder = new PreparedStatementConditionBuilder();
        $conditionBuilder->add("boardID IN (?)", [$boardIDs]);
        $conditionBuilder->add("status = ?", [$status]);

        $sql = "DELETE FROM     wbb1_board_user_status
                {$conditionBuilder}";
        $statement = WCF::getDB()->prepare($sql);
        $statement->execute($conditionBuilder->getParameters());

        UserStorageHandler::getInstance()->resetAll('wbbUnreadWatchedBoards');
        UserStorageHandler::getInstance()->resetAll('wbbWatchedBoards');
    }

    /**
     * Deletes all subscriptions with the given status ('threadsAndPosts', 'ignoring' or 'threadsAndPosts') for a specific user.
     *
     * @throws InvalidArgumentException If the status is not 'threadsAndPosts', 'threadsAndPosts' nor 'ignoring'.
     * @throws InvalidArgumentException If the given user is a guest.
     */
    public function deleteAllSubscriptionsForUser(User $user, string $status): void
    {
        if (!$user->userID) {
            throw new \InvalidArgumentException(
                "Deleting status for guests is unsupported."
            );
        }

        if (
            !\in_array($status, [
                self::SUBSCRIPTION_MODE_THREADSONLY,
                self::SUBSCRIPTION_MODE_IGNORING,
                self::SUBSCRIPTION_MODE_THREADSANDPOSTS,
            ])
        ) {
            throw new \InvalidArgumentException(\sprintf(
                "Unknown board status '%s' given.",
                $status
            ));
        }

        $conditionBuilder = new PreparedStatementConditionBuilder();
        $conditionBuilder->add("userID = ?", [$user->userID]);
        $conditionBuilder->add("status = ?", [$status]);

        $sql = "DELETE FROM     wbb1_board_user_status
                {$conditionBuilder}";
        $statement = WCF::getDB()->prepare($sql);
        $statement->execute($conditionBuilder->getParameters());

        UserStorageHandler::getInstance()->reset([$user->userID], 'wbbUnreadWatchedBoards');
        UserStorageHandler::getInstance()->reset([$user->userID], 'wbbWatchedBoards');
    }

    /**
     * Returns a list of userIDs which have subscribed (threadsAndPosts) or threadsAndPosts the given board.
     *
     * @return int[]
     */
    public function getSubscribers(int $boardID): array
    {
        return \array_merge(
            $this->getUserIDsForStatus($boardID, self::SUBSCRIPTION_MODE_THREADSONLY),
            $this->getUserIDsForStatus($boardID, self::SUBSCRIPTION_MODE_THREADSANDPOSTS)
        );
    }

    /**
     * Returns a list of userIDs which have subscribed the given board for threads and posts.
     *
     * @return int[]
     */
    public function getThreadAndPostSubscribers(int $boardID): array
    {
        return $this->getUserIDsForStatus($boardID, self::SUBSCRIPTION_MODE_THREADSANDPOSTS);
    }

    /**
     * Filters out the userIDs that ignore the given board from the given userIDs and returns
     * only the userIDs that do not ignore the topic.
     *
     * @return int[]
     */
    public function filterIgnoredUserIDs(array $userIDs, int $boardID): array
    {
        return \array_diff(
            $userIDs,
            $this->getUserIDsForStatus($boardID, self::SUBSCRIPTION_MODE_IGNORING)
        );
    }

    /**
     * Returns all boardIDs that are subscribed with the given status by the given user.
     *
     * @throws InvalidArgumentException If the status is not 'threadsAndPosts', 'threadsAndPosts' nor 'ignoring'.
     * @throws InvalidArgumentException If the given user is a guest.
     * @return int[]
     */
    public function getBoardIDsForStatus(string $status, ?User $user = null): array
    {
        if ($user === null) {
            $user = WCF::getUser();
        }

        if (!$user->userID) {
            throw new \InvalidArgumentException(
                "Fetching status for guests is unsupported."
            );
        }

        if (
            !\in_array($status, [
                self::SUBSCRIPTION_MODE_THREADSONLY,
                self::SUBSCRIPTION_MODE_IGNORING,
                self::SUBSCRIPTION_MODE_THREADSANDPOSTS,
            ])
        ) {
            throw new \InvalidArgumentException(\sprintf(
                "Unknown board status '%s' given.",
                $status
            ));
        }

        $sql = "SELECT      boardID
                FROM        wbb1_board_user_status
                WHERE       userID = ?
                        AND status = ?";
        $statement = WCF::getDB()->prepare($sql);
        $statement->execute([
            $user->userID,
            $status,
        ]);

        $boardIDs = $statement->fetchAll(\PDO::FETCH_COLUMN);

        foreach ($boardIDs as $boardID) {
            $this->statusCache[$user->userID][$boardID] = $status;
        }

        return $boardIDs;
    }

    /**
     * Fires the thread notification for a specific thread.
     */
    public function fireThreadNotification(Thread $thread): void
    {
        $userIDs = $this->getSubscribers($thread->boardID);

        $userIDs = ThreadStatusHandler::filterIgnoredUserIDs($userIDs, $thread->threadID);

        $userIDs = \array_diff(
            $userIDs,
            [$thread->userID]
        );

        if (empty($userIDs)) {
            return;
        }

        UserNotificationHandler::getInstance()->fireEvent(
            'thread',
            'com.woltlab.wbb.thread',
            new ThreadUserNotificationObject($thread),
            $userIDs,
            [],
            0,
            $thread->languageID
        );

        UserStorageHandler::getInstance()->reset($userIDs, 'wbbUnreadWatchedBoards');
    }

    private function getUserIDsForStatus(int $boardID, string $status): array
    {
        if ($status === self::SUBSCRIPTION_MODE_NORMAL) {
            throw new \InvalidArgumentException("Fetching users with status 'normal' is not supported.");
        }

        if (!isset($this->statusBoardCache[$boardID][$status])) {
            $sql = "SELECT      userID
                    FROM        wbb1_board_user_status
                    WHERE       boardID = ?
                            AND status = ?";
            $statement = WCF::getDB()->prepare($sql);
            $statement->execute([
                $boardID,
                $status,
            ]);

            $this->statusBoardCache[$boardID][$status] = $statement->fetchAll(\PDO::FETCH_COLUMN);
        }

        return $this->statusBoardCache[$boardID][$status];
    }
}
